#!/usr/bin/env python3
from decimal import Decimal, getcontext

getcontext().prec = 200

class PhiCompressor:
    PHI = Decimal('1.61803398874989484820458683436563811772030917980576')

    def __init__(self, max_exponent=500):
        self.max_exponent = max_exponent
        # Precompute phi powers for exact discrete folding
        self.phi_powers = [self.PHI ** i for i in range(max_exponent + 1)]

    def fold_sequence(self, seq):
        """Fold a single integer sequence into discrete phi-powers"""
        residual = sum(seq)
        folded = []
        for i in reversed(range(self.max_exponent + 1)):
            power = self.phi_powers[i]
            while residual >= power:
                folded.append(i)
                residual -= power
        return folded

    def decode_folded(self, folded):
        """Reconstruct original sequence from folded phi-powers"""
        return [int(sum(self.phi_powers[e] for e in folded))]

def main():
    inp = input("Enter sequences: ")
    sequences = [list(map(int, s.split(','))) for s in inp.strip().split()]
    compressor = PhiCompressor(max_exponent=200)

    folded_all = [compressor.fold_sequence(seq) for seq in sequences]
    print("Folded exponents:", folded_all)

    reconstructed = [compressor.decode_folded(folded) for folded in folded_all]
    print("Reconstructed sequences:", reconstructed)

if __name__ == "__main__":
    main()
